<?php
session_start();
include('../config/db_connect.php');

// Redirect if not logged in
if (!isset($_SESSION['hod_id'])) {
    header("Location: hod_login.php");
    exit;
}

$hod_id   = (int)$_SESSION['hod_id'];
$hod_name = $_SESSION['hod_name'] ?? '';
$dept_id  = $_SESSION['hod_department'] ?? null;

// Fetch department name
$dept_name = "";
if ($dept_id) {
    $stmt = $conn->prepare("SELECT dept_name FROM departments WHERE dept_id = ? LIMIT 1");
    $stmt->bind_param("i", $dept_id);
    $stmt->execute();
    $res = $stmt->get_result()->fetch_assoc();
    $dept_name = $res['dept_name'] ?? '';
}

$message = "";
$errors  = [];

// -----------------------------
// PROCESS UPLOAD
// -----------------------------
if ($_SERVER['REQUEST_METHOD'] === "POST" && isset($_POST['upload_tests'])) {

    $subject      = trim($_POST['subject']);
    $teacher_name = trim($_POST['teacher_name']);
    $term         = trim($_POST['term']);

    if (!$subject || !$teacher_name || !$term) {
        $errors[] = "All fields are required.";
    }

    if (!isset($_FILES['files'])) {
        $errors[] = "Please select at least one file.";
    }

    if (empty($errors)) {
        $files = $_FILES['files'];
        $total = count($files['name']);

        if ($total > 10) {
            $errors[] = "You can upload up to 10 files.";
        } else {
            $folder = "../uploads/hod_$hod_id";
            if (!is_dir($folder)) mkdir($folder, 0755, true);

            $uploaded = 0;

            for ($i = 0; $i < $total; $i++) {

                if ($files['error'][$i] !== UPLOAD_ERR_OK) continue;

                $original = basename($files['name'][$i]);
                $tmp      = $files['tmp_name'][$i];
                $size     = $files['size'][$i];

                if ($size > (5 * 1024 * 1024)) {
                    $errors[] = "$original is too large (max 5MB).";
                    continue;
                }

                $ext = strtolower(pathinfo($original, PATHINFO_EXTENSION));
                if (!in_array($ext, ['pdf', 'docx'])) {
                    $errors[] = "$original is not a PDF or DOCX.";
                    continue;
                }

                $new_name = time() . "_" . bin2hex(random_bytes(4)) . ".$ext";
                $save_path = "$folder/$new_name";

                if (move_uploaded_file($tmp, $save_path)) {

                    $stmt = $conn->prepare("
                        INSERT INTO hod_uploads 
                        (hod_id, department, term, subject, teacher_name, file_name, original_name) 
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    ");

                    $relative = "uploads/hod_$hod_id/$new_name";

                    $stmt->bind_param("issssss",
                        $hod_id,
                        $dept_name,
                        $term,
                        $subject,
                        $teacher_name,
                        $relative,
                        $original
                    );

                    $stmt->execute();
                    $stmt->close();

                    $uploaded++;

                } else {
                    $errors[] = "$original failed to upload.";
                }
            }

            if ($uploaded > 0) $message = "$uploaded file(s) uploaded successfully.";
            else if (empty($errors)) $errors[] = "No files were uploaded.";
        }
    }
}

// -----------------------------
// Fetch uploads
// -----------------------------
$res = $conn->query("
    SELECT * FROM hod_uploads 
    WHERE hod_id = $hod_id 
    ORDER BY upload_date DESC
");

?>
<!DOCTYPE html>
<html>
<head>
    <title>Upload Tests</title>
    <link rel="stylesheet" href="css/hod_style.css">
</head>
<body>

<div class="container">

    <div class="topbar">
        <h2>Upload Tests</h2>
        <div>
            <a href="hod_dashboard.php">Dashboard</a>
            <a href="hod_logout.php">Logout</a>
        </div>
    </div>

    <div class="grid">

        <div>
            <div class="card">
                <h3>New Upload</h3>

                <?php if ($message): ?>
                    <p class="success"><?= $message ?></p>
                <?php endif; ?>

                <?php foreach ($errors as $e): ?>
                    <p class="error"><?= $e ?></p>
                <?php endforeach; ?>

                <form method="post" enctype="multipart/form-data">

                    <label>Subject / Test Name</label>
                    <input type="text" name="subject" required>

                    <label>Teacher</label>
                    <input type="text" name="teacher_name" required>

                    <label>Term</label>
                    <select name="term" required>
                        <option value="">-- Select Term --</option>
                        <option>Term 1</option>
                        <option>Term 2</option>
                        <option>Term 3</option>
                    </select>

                    <label>Select Files (PDF/DOCX)</label>
                    <input type="file" name="files[]" multiple required accept=".pdf,.docx">

                    <button type="submit" name="upload_tests">Upload</button>
                </form>
            </div>

            <div class="card">
                <h3>Your Uploads</h3>

                <?php if ($res->num_rows == 0): ?>
                    <p>No uploads yet.</p>
                <?php else: ?>
                    <?php while ($row = $res->fetch_assoc()): ?>
                        <div class="item">
                            <div>
                                <strong><?= $row['subject'] ?></strong><br>
                                <span><?= $row['original_name'] ?></span><br>
                                <small><?= $row['upload_date'] ?></small>
                            </div>
                            <div>
                                <a href="hod_download.php?id=<?= $row['upload_id'] ?>">Download</a>
                                &nbsp;|&nbsp;
                                <a href="delete_upload.php?id=<?= $row['upload_id'] ?>"
                                   onclick="return confirm('Delete this file?')">
                                   Delete
                                </a>
                            </div>
                        </div>
                    <?php endwhile; ?>
                <?php endif; ?>

            </div>
        </div>

        <div>
            <div class="card">
                <h3>Department</h3>
                <p><?= $dept_name ?></p>
            </div>
        </div>

    </div>

</div>

</body>
</html>
